package furny.swing.admin.tags;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import furny.entities.Tag;
import furny.entities.TagType;
import furny.furndb.FurnDBManager;
import furny.furndb.TagUpdateListener;

/**
 * {@link JTree} that displays tags grouped by their type.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class TagTree extends JTree implements TagUpdateListener {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(TagTree.class.getName());

  private final TagTreeModel model = new TagTreeModel();

  private List<Tag> allTags = new ArrayList<Tag>();

  /**
   * Instantiates a new tag tree.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public TagTree() {
    setModel(model);

    setRootVisible(false);

    final DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
    renderer.setOpenIcon(null);
    renderer.setClosedIcon(null);
    renderer.setLeafIcon(null);
    setCellRenderer(renderer);

    setTransferHandler(new TagTransferHandler(this));
    setDragEnabled(true);

    FurnDBManager.getInstance().addTagUpdateListener(this);

    update();
  }

  /**
   * Gets the selected tags.
   * 
   * @return the selected tags
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public List<Tag> getSelectedTags() {
    final List<Tag> tags = new ArrayList<Tag>();

    if (getSelectionPaths() != null) {
      for (final TreePath tp : getSelectionPaths()) {
        final Object o = tp.getLastPathComponent();
        if (o instanceof TagNode) {
          tags.add(((TagNode) o).getTag());
        }
      }
    }

    return tags;
  }

  /**
   * Gets the all tags.
   * 
   * @return the all tags
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public List<Tag> getAllTags() {
    return allTags;
  }

  /**
   * Updates the tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void update() {
    FurnDBManager.getInstance().updateTags(null);
  }

  @Override
  public void tagsUpdated(final TagType type, final List<Tag> tags) {
    if (type == null) {
      SwingUtilities.invokeLater(new Runnable() {
        @Override
        public void run() {
          allTags = tags;
          Collections.sort(allTags);

          final DefaultMutableTreeNode root = new DefaultMutableTreeNode("Tags");

          final TagType[] tagTypes = TagType.values();
          Arrays.sort(tagTypes, new Comparator<TagType>() {
            @Override
            public int compare(final TagType o1, final TagType o2) {
              return o2.getRanking() - o1.getRanking();
            }
          });

          for (final TagType type : tagTypes) {

            final TagTypeNode n = new TagTypeNode(type);
            root.add(n);
            for (final Tag tag : tags) {
              if (type.equals(tag.getType())) {
                n.add(new TagNode(tag));
              }
            }
          }

          model.setRoot(root);

          model.fireTreeStructureChanged(root);

          for (int i = 0; i < getRowCount(); i++) {
            expandRow(i);
          }
        }
      });
    }
  }

  /**
   * Tree model for tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static class TagTreeModel extends DefaultTreeModel {

    /**
     * Instantiates a new tag tree model.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public TagTreeModel() {
      super(new DefaultMutableTreeNode("Tags"));
    }

    @Override
    public DefaultMutableTreeNode getRoot() {
      return (DefaultMutableTreeNode) super.getRoot();
    }

    /**
     * Refreshs all nodes.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void refreshAllNodes() {
      refreshNode(getRoot());
    }

    /**
     * Refresh a node.
     * 
     * @param tn
     *          the node to refresh.
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    private void refreshNode(final TreeNode tn) {
      fireTreeNodesChanged(tn);
      for (int i = 0; i < tn.getChildCount(); i++) {
        refreshNode(tn.getChildAt(i));
      }
    }

    /**
     * Fires tree structure changed event.
     * 
     * @param node
     *          the node
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void fireTreeStructureChanged(final TreeNode node) {
      try {
        final TreeModelEvent event = new TreeModelEvent(this,
            getPathToRoot(node));
        for (final TreeModelListener l : getTreeModelListeners()) {
          l.treeStructureChanged(event);
        }
      } catch (final NullPointerException e) {
        LOGGER.log(Level.WARNING, "An error occured", e);
      }
    }

    /**
     * Fires tree nodes changed event.
     * 
     * @param node
     *          the node
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void fireTreeNodesChanged(final TreeNode node) {
      try {
        final TreeModelEvent event = new TreeModelEvent(this,
            getPathToRoot(node));
        for (final TreeModelListener l : getTreeModelListeners()) {
          l.treeNodesChanged(event);
        }
      } catch (final NullPointerException e) {
        LOGGER.log(Level.WARNING, "An error occured", e);
      }
    }
  }

  /**
   * Tree node for a tag type.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static class TagTypeNode extends DefaultMutableTreeNode {
    private final TagType type;

    /**
     * Instantiates a new tag type node.
     * 
     * @param type
     *          the type
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public TagTypeNode(final TagType type) {
      this.type = type;
    }

    @Override
    public String toString() {
      return type.name();
    }

    /**
     * Gets the type.
     * 
     * @return the type
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public TagType getType() {
      return type;
    }
  }

  /**
   * Tree node for a tag.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static class TagNode extends DefaultMutableTreeNode {
    private final Tag tag;

    /**
     * Instantiates a new tag node.
     * 
     * @param tag
     *          the tag
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public TagNode(final Tag tag) {
      this.tag = tag;
    }

    @Override
    public String toString() {
      return tag.getName();
    }

    /**
     * Gets the tag.
     * 
     * @return the tag
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public Tag getTag() {
      return tag;
    }
  }
}
